'use client'

import { motion } from 'framer-motion'
import { 
  GlobeAltIcon, 
  CloudIcon, 
  MegaphoneIcon, 
  CodeBracketIcon,
  ChartBarIcon,
  MagnifyingGlassIcon,
  ShareIcon,
  DevicePhoneMobileIcon
} from '@heroicons/react/24/outline'
import { formatPrice } from '@/lib/utils'
import { useHydrationSafeCart } from '@/hooks/useHydrationSafeCart'

const services = [
  {
    id: '1',
    name: 'Starter Hosting',
    category: 'HOSTING' as const,
    description: 'Perfect for personal websites and small blogs with essential hosting features.',
    shortDescription: 'Perfect for personal websites',
    price: 5.99,
    pricingModel: 'MONTHLY' as const,
    icon: GlobeAltIcon,
    features: [
      '1 Website',
      '10 GB SSD Storage',
      '100 GB Bandwidth',
      'Free SSL Certificate',
      'Email Accounts (5)',
      'cPanel Access',
      '24/7 Support'
    ],
    popular: false,
    color: 'bg-gray-500',
    gradient: 'from-gray-100 to-gray-200',
    accentColor: 'gray'
  },
  {
    id: '2',
    name: 'Business Hosting',
    category: 'HOSTING' as const,
    description: 'Ideal for growing businesses with enhanced performance and advanced features.',
    shortDescription: 'Ideal for growing businesses',
    price: 12.99,
    pricingModel: 'MONTHLY' as const,
    icon: CloudIcon,
    features: [
      '5 Websites',
      '50 GB SSD Storage',
      'Unlimited Bandwidth',
      'Free SSL Certificate',
      'Unlimited Email Accounts',
      'cPanel + WordPress',
      'Daily Backups',
      '99.9% Uptime Guarantee'
    ],
    popular: true,
    color: 'bg-blue-500',
    gradient: 'from-blue-100 to-blue-200',
    accentColor: 'blue'
  },
  {
    id: '3',
    name: 'Professional Hosting',
    category: 'HOSTING' as const,
    description: 'Advanced hosting solution for established businesses requiring premium performance.',
    shortDescription: 'Premium performance hosting',
    price: 24.99,
    pricingModel: 'MONTHLY' as const,
    icon: MegaphoneIcon,
    features: [
      'Unlimited Websites',
      '100 GB SSD Storage',
      'Unlimited Bandwidth',
      'Free SSL + CDN',
      'Priority Support',
      'Advanced Security',
      'WordPress Optimization',
      'Free Domain (.com)'
    ],
    popular: false,
    color: 'bg-purple-500',
    gradient: 'from-purple-100 to-purple-200',
    accentColor: 'purple'
  },
  {
    id: '4',
    name: 'Enterprise Hosting',
    category: 'HOSTING' as const,
    description: 'Enterprise-grade hosting with dedicated resources and premium support.',
    shortDescription: 'Enterprise-grade performance',
    price: 49.99,
    pricingModel: 'MONTHLY' as const,
    icon: CodeBracketIcon,
    features: [
      'Unlimited Everything',
      '200 GB SSD Storage',
      'Dedicated IP',
      'Advanced SSL + CDN',
      'White-label Options',
      'Priority Phone Support',
      'Server Monitoring',
      'Custom Configurations'
    ],
    popular: false,
    color: 'bg-green-500',
    gradient: 'from-green-100 to-green-200',
    accentColor: 'green'
  },
  // Additional Services
  {
    id: '5',
    name: 'Domain Registration',
    category: 'DOMAINS' as const,
    description: 'Secure your perfect domain name with competitive pricing and reliable management.',
    shortDescription: 'Get your perfect domain name',
    price: 12.99,
    pricingModel: 'YEARLY' as const,
    icon: GlobeAltIcon,
    features: [
      'Free WHOIS protection',
      'Easy domain management',
      'Auto-renewal options',
      'DNS management tools',
      '24/7 support'
    ],
    popular: false,
    color: 'bg-indigo-500',
    gradient: 'from-indigo-100 to-indigo-200',
    accentColor: 'indigo'
  },
  {
    id: '6',
    name: 'Digital Marketing',
    category: 'MARKETING' as const,
    description: 'Comprehensive digital marketing strategies to boost your online presence and ROI.',
    shortDescription: 'Grow your online presence',
    price: 499.99,
    pricingModel: 'MONTHLY' as const,
    icon: MegaphoneIcon,
    features: [
      'Social media management',
      'Content creation',
      'Campaign optimization',
      'Analytics reporting',
      'Strategy consultation'
    ],
    popular: false,
    color: 'bg-pink-500',
    gradient: 'from-pink-100 to-pink-200',
    accentColor: 'pink'
  },
  {
    id: '7',
    name: 'Software Development',
    category: 'DEVELOPMENT' as const,
    description: 'Custom software solutions tailored to your business needs with modern technologies.',
    shortDescription: 'Custom software solutions',
    price: 2999.99,
    pricingModel: 'ONE_TIME' as const,
    icon: CodeBracketIcon,
    features: [
      'Custom development',
      'Modern frameworks',
      'Mobile responsive',
      'Testing included',
      'Maintenance support'
    ],
    popular: false,
    color: 'bg-teal-500',
    gradient: 'from-teal-100 to-teal-200',
    accentColor: 'teal'
  },
  {
    id: '8',
    name: 'Google Ads Management',
    category: 'MARKETING' as const,
    description: 'Professional Google Ads campaigns that deliver results and maximize your advertising ROI.',
    shortDescription: 'Maximize your ad ROI',
    price: 299.99,
    pricingModel: 'MONTHLY' as const,
    icon: ChartBarIcon,
    features: [
      'Campaign setup',
      'Keyword optimization',
      'Ad copy creation',
      'Performance tracking',
      'Monthly reports'
    ],
    popular: true,
    color: 'bg-yellow-500',
    gradient: 'from-yellow-100 to-yellow-200',
    accentColor: 'yellow'
  }
]

const containerVariants = {
  hidden: { opacity: 0 },
  visible: {
    opacity: 1,
    transition: {
      staggerChildren: 0.1
    }
  }
}

const itemVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: {
    opacity: 1,
    y: 0,
    transition: { duration: 0.5 }
  }
}

export default function Services() {
  const { addItem, openCart } = useHydrationSafeCart()

  const handleAddToCart = (service: typeof services[0]) => {
    addItem({
      serviceId: service.id,
      name: service.name,
      price: service.price,
      quantity: 1,
    })
    setTimeout(() => openCart(), 100)
  }

  return (
    <section id="services" className="py-20 bg-gradient-to-b from-white via-blue-50/30 to-purple-50/30">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Section Header */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5 }}
          className="text-center mb-14"
        >
          <div className="inline-flex items-center px-4 py-2 bg-blue-50 border border-blue-200 rounded-lg mb-6">
            <span className="text-sm font-semibold text-blue-900">Premium Solutions</span>
          </div>
          <h2 className="text-4xl md:text-5xl font-bold text-slate-900 mb-4">
            Skyrocket Your Business With <span className="text-blue-600">Premium Solutions</span>
          </h2>
          <p className="text-lg text-slate-600 max-w-2xl mx-auto">
            Step into the future with our powerful cloud & hosting, digital marketing, domains, software development, & more. Boost sales, security, and business growth.
          </p>
        </motion.div>

        {/* Main Service Cards (3) */}
        <motion.div
          variants={containerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-12"
        >
          {services.slice(0, 3).map((service, index) => {
            const Icon = service.icon
            return (
              <motion.div
                key={service.id}
                variants={itemVariants}
                className={`relative bg-white rounded-2xl border border-slate-200 shadow-lg hover:shadow-2xl transition-all duration-300 overflow-hidden group ${service.popular ? 'ring-2 ring-blue-500' : ''}`}
              >
                {/* Popular badge */}
                {service.popular && (
                  <div className="absolute top-0 right-0 bg-blue-600 text-white px-4 py-1 rounded-bl-2xl text-xs font-bold z-10">
                    Most Popular
                  </div>
                )}
                <div className="p-8 flex flex-col items-center">
                  {/* Icon */}
                  <div className="w-12 h-12 bg-blue-50 rounded-xl flex items-center justify-center mb-4">
                    <Icon className="w-6 h-6 text-blue-600" />
                  </div>
                  {/* Content */}
                  <h3 className="text-2xl font-bold text-slate-900 mb-2 text-center">
                    {service.name}
                  </h3>
                  <p className="text-slate-600 mb-4 text-base text-center">
                    {service.shortDescription}
                  </p>
                  {/* Features */}
                  <ul className="space-y-2 mb-6">
                    {service.features.slice(0, 4).map((feature, featureIndex) => (
                      <li 
                        key={featureIndex} 
                        className="text-sm text-slate-700 flex items-center justify-center"
                      >
                        <div className="w-1.5 h-1.5 bg-blue-600 rounded-full mr-2 flex-shrink-0" />
                        <span>{feature}</span>
                      </li>
                    ))}
                  </ul>
                  {/* Pricing and CTA */}
                  <div className="pt-4 border-t border-slate-200 w-full">
                    <div className="mb-4 text-center">
                      <div className="flex items-baseline justify-center">
                        <span className="text-3xl font-bold text-slate-900">
                          {formatPrice(service.price)}
                        </span>
                        <span className="text-slate-500 ml-1 text-sm">
                          /{service.pricingModel.toLowerCase()}
                        </span>
                      </div>
                    </div>
                    <button
                      onClick={() => handleAddToCart(service)}
                      className="w-full bg-blue-600 hover:bg-blue-700 text-white py-3 px-4 rounded-lg font-semibold transition-colors duration-200 shadow-md"
                    >
                      Choose Plan
                    </button>
                  </div>
                </div>
              </motion.div>
            )
          })}
        </motion.div>

        {/* Additional Services Row */}
        <motion.div
          variants={containerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true }}
          className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-10"
        >
          {services.slice(3, 7).map((service) => {
            const Icon = service.icon
            return (
              <motion.div
                key={service.id}
                variants={itemVariants}
                className="bg-white rounded-xl p-6 shadow-sm hover:shadow-lg transition-all duration-300 border border-slate-100 flex flex-col items-center"
              >
                <div className={`w-10 h-10 ${service.color} rounded-lg flex items-center justify-center mb-3`}>
                  <Icon className="w-5 h-5 text-white" />
                </div>
                <h4 className="text-lg font-semibold text-slate-900 mb-1 text-center">
                  {service.name}
                </h4>
                <p className="text-slate-600 mb-2 text-sm text-center">
                  {service.shortDescription}
                </p>
                <ul className="space-y-1 mb-4">
                  {service.features.slice(0, 2).map((feature, index) => (
                    <li key={index} className="text-xs text-slate-600 flex items-center justify-center">
                      <div className="w-1.5 h-1.5 bg-blue-600 rounded-full mr-2 flex-shrink-0" />
                      <span>{feature}</span>
                    </li>
                  ))}
                </ul>
                <div className="mb-2 text-center">
                  <span className="text-base font-bold text-slate-900">
                    {formatPrice(service.price)}
                  </span>
                  <span className="text-slate-500 ml-1 text-xs">
                    /{service.pricingModel.toLowerCase()}
                  </span>
                </div>
                <button
                  onClick={() => handleAddToCart(service)}
                  className="w-full bg-blue-50 hover:bg-blue-100 text-blue-700 py-2 px-3 rounded-lg font-medium transition-colors duration-200 text-xs"
                >
                  Add to Cart
                </button>
              </motion.div>
            )
          })}
        </motion.div>
      </div>
    </section>
  )
}