import { NextAuthOptions } from 'next-auth'
import { PrismaAdapter } from '@auth/prisma-adapter'
import GoogleProvider from 'next-auth/providers/google'
import GitHubProvider from 'next-auth/providers/github'
import CredentialsProvider from 'next-auth/providers/credentials'
import bcrypt from 'bcryptjs'
import { prisma } from './prisma'
import { UserRole } from '@prisma/client'

export const authOptions: NextAuthOptions = {
  adapter: PrismaAdapter(prisma) as any,
  providers: [
    GoogleProvider({
      clientId: process.env.GOOGLE_CLIENT_ID!,
      clientSecret: process.env.GOOGLE_CLIENT_SECRET!,
    }),
    GitHubProvider({
      clientId: process.env.GITHUB_CLIENT_ID!,
      clientSecret: process.env.GITHUB_CLIENT_SECRET!,
    }),
    CredentialsProvider({
      name: 'credentials',
      credentials: {
        email: { label: 'Email', type: 'email' },
        password: { label: 'Password', type: 'password' }
      },
      async authorize(credentials) {
        try {
          console.log('[auth] authorize: received credentials:', credentials);
          if (!credentials?.email || !credentials?.password) {
            console.log('[auth] authorize: missing credentials', credentials);
            return null;
          }

          // Log incoming email (do NOT log password)
          console.log('[auth] authorize: request for email:', credentials.email);

          const user = await prisma.user.findUnique({
            where: {
              email: credentials.email
            }
          });

          console.log('[auth] authorize: user lookup result:', user);

          if (!user) {
            console.log('[auth] authorize: user not found for', credentials.email);
            return null;
          }

          if (!user.password) {
            console.log('[auth] authorize: user has no password stored', credentials.email);
            return null;
          }

          console.log('[auth] authorize: comparing password for', credentials.email);
          const isPasswordValid = await bcrypt.compare(
            credentials.password,
            user.password
          );

          console.log('[auth] authorize: password compare for', credentials.email, '=>', isPasswordValid);

          if (!isPasswordValid) {
            console.log('[auth] authorize: invalid password for', credentials.email);
            return null;
          }

          const resultUser = {
            id: user.id,
            email: user.email,
            name: user.name,
            role: user.role,
            image: user.image,
          };
          console.log('[auth] authorize: returning user object', resultUser);
          return resultUser;
        } catch (err) {
          console.error('[auth] authorize error', err);
          return null;
        }
      }
    })
  ],
  session: {
    strategy: 'jwt'
  },
  callbacks: {
    async jwt({ token, user }) {
      if (user) {
        token.role = user.role || UserRole.CLIENT
      }
      return token
    },
    async session({ session, token }) {
      if (token) {
        session.user.id = token.sub!
        session.user.role = token.role as UserRole
      }
      return session
    },
    async signIn({ user, account, profile }) {
      if (account?.provider === 'google' || account?.provider === 'github') {
        // For OAuth providers, ensure user gets CLIENT role by default
        if (!user.role) {
          await prisma.user.update({
            where: { email: user.email! },
            data: { role: UserRole.CLIENT }
          })
        }
      }
      return true
    }
  },
  pages: {
    signIn: '/auth/signin',
  },
  secret: process.env.NEXTAUTH_SECRET,
}