// Production configuration and environment validation
export const productionConfig = {
  // Server configuration
  server: {
    port: parseInt(process.env.PORT || '3000'),
    host: process.env.HOST || '0.0.0.0',
    nodeEnv: process.env.NODE_ENV || 'development',
    trustProxy: process.env.TRUST_PROXY === 'true',
  },

  // Database configuration
  database: {
    url: process.env.DATABASE_URL,
    maxConnections: parseInt(process.env.DB_MAX_CONNECTIONS || '10'),
    connectionTimeout: parseInt(process.env.DB_CONNECTION_TIMEOUT || '30000'),
    queryTimeout: parseInt(process.env.DB_QUERY_TIMEOUT || '15000'),
    enableLogging: process.env.DB_ENABLE_LOGGING === 'true',
    ssl: process.env.NODE_ENV === 'production' ? { rejectUnauthorized: false } : false,
  },

  // Authentication configuration
  auth: {
    secret: process.env.NEXTAUTH_SECRET,
    url: process.env.NEXTAUTH_URL,
    providers: {
      credentials: {
        enabled: true,
        maxAttempts: 5,
        lockoutDuration: 15 * 60 * 1000, // 15 minutes
      },
    },
    session: {
      strategy: 'jwt' as const,
      maxAge: 7 * 24 * 60 * 60, // 7 days
      updateAge: 24 * 60 * 60, // 1 day
    },
    pages: {
      signIn: '/auth/signin',
      signUp: '/auth/signup',
      error: '/auth/error',
    },
  },

  // Payment configuration
  payment: {
    stripe: {
      publishableKey: process.env.NEXT_PUBLIC_STRIPE_PUBLISHABLE_KEY,
      secretKey: process.env.STRIPE_SECRET_KEY,
      webhookSecret: process.env.STRIPE_WEBHOOK_SECRET,
      apiVersion: '2023-10-16' as const,
    },
    currency: 'usd',
    allowedCountries: ['US', 'CA', 'GB', 'AU', 'DE', 'FR', 'IT', 'ES', 'NL', 'BE'],
  },

  // Email configuration
  email: {
    from: process.env.EMAIL_FROM || 'noreply@nuwairsystems.com',
    smtp: {
      host: process.env.SMTP_HOST,
      port: parseInt(process.env.SMTP_PORT || '587'),
      secure: process.env.SMTP_SECURE === 'true',
      auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASS,
      },
    },
    templates: {
      welcome: 'welcome.html',
      passwordReset: 'password-reset.html',
      orderConfirmation: 'order-confirmation.html',
      invoice: 'invoice.html',
    },
  },

  // File upload configuration
  upload: {
    maxSize: 10 * 1024 * 1024, // 10MB
    allowedTypes: [
      'image/jpeg',
      'image/png',
      'image/webp',
      'image/svg+xml',
      'application/pdf',
      'text/plain',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    ],
    destination: process.env.UPLOAD_DESTINATION || './uploads',
    useCloudStorage: process.env.USE_CLOUD_STORAGE === 'true',
    cloudStorage: {
      provider: process.env.CLOUD_STORAGE_PROVIDER || 'aws', // aws, gcp, azure
      bucket: process.env.CLOUD_STORAGE_BUCKET,
      region: process.env.CLOUD_STORAGE_REGION,
      accessKeyId: process.env.CLOUD_STORAGE_ACCESS_KEY,
      secretAccessKey: process.env.CLOUD_STORAGE_SECRET_KEY,
    },
  },

  // Caching configuration
  cache: {
    enabled: process.env.CACHE_ENABLED !== 'false',
    provider: process.env.CACHE_PROVIDER || 'memory', // memory, redis, memcached
    redis: {
      host: process.env.REDIS_HOST || 'localhost',
      port: parseInt(process.env.REDIS_PORT || '6379'),
      password: process.env.REDIS_PASSWORD,
      db: parseInt(process.env.REDIS_DB || '0'),
      maxRetries: 3,
      retryDelayOnFailover: 100,
    },
    ttl: {
      default: 5 * 60 * 1000, // 5 minutes
      short: 1 * 60 * 1000, // 1 minute
      medium: 15 * 60 * 1000, // 15 minutes
      long: 60 * 60 * 1000, // 1 hour
      static: 24 * 60 * 60 * 1000, // 24 hours
    },
  },

  // Monitoring and logging
  monitoring: {
    enabled: process.env.MONITORING_ENABLED !== 'false',
    logLevel: process.env.LOG_LEVEL || (process.env.NODE_ENV === 'production' ? 'warn' : 'debug'),
    enableRequestLogging: process.env.ENABLE_REQUEST_LOGGING === 'true',
    enableErrorTracking: process.env.ENABLE_ERROR_TRACKING !== 'false',
    externalServices: {
      datadog: {
        apiKey: process.env.DATADOG_API_KEY,
        applicationKey: process.env.DATADOG_APPLICATION_KEY,
        site: process.env.DATADOG_SITE || 'datadoghq.com',
      },
      sentry: {
        dsn: process.env.SENTRY_DSN,
        environment: process.env.NODE_ENV,
        release: process.env.npm_package_version,
      },
      newRelic: {
        licenseKey: process.env.NEW_RELIC_LICENSE_KEY,
        appName: process.env.NEW_RELIC_APP_NAME || 'NuwairSystems',
      },
    },
  },

  // Rate limiting configuration
  rateLimit: {
    enabled: process.env.RATE_LIMIT_ENABLED !== 'false',
    store: process.env.RATE_LIMIT_STORE || 'memory', // memory, redis
    skipSuccessfulRequests: false,
    skipFailedRequests: false,
    standardHeaders: true,
    legacyHeaders: false,
    limits: {
      global: { requests: 1000, window: '15m' },
      api: { requests: 100, window: '15m' },
      auth: { requests: 5, window: '15m' },
      payment: { requests: 10, window: '15m' },
      upload: { requests: 20, window: '15m' },
    },
  },

  // CORS configuration
  cors: {
    enabled: process.env.CORS_ENABLED !== 'false',
    origin: process.env.CORS_ORIGIN?.split(',') || [
      process.env.NODE_ENV === 'production'
        ? 'https://nuwairsystems.com'
        : 'http://localhost:3000'
    ],
    credentials: true,
    maxAge: 86400, // 24 hours
    allowedMethods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],
    allowedHeaders: [
      'Origin',
      'X-Requested-With',
      'Content-Type',
      'Accept',
      'Authorization',
      'X-CSRF-Token',
    ],
  },

  // CDN and static assets
  cdn: {
    enabled: process.env.CDN_ENABLED === 'true',
    baseUrl: process.env.CDN_BASE_URL,
    imageOptimization: {
      enabled: process.env.IMAGE_OPTIMIZATION_ENABLED !== 'false',
      quality: parseInt(process.env.IMAGE_QUALITY || '85'),
      formats: ['webp', 'avif', 'jpeg'],
      sizes: [640, 768, 1024, 1280, 1920],
    },
  },

  // Feature flags
  features: {
    enableRegistration: process.env.ENABLE_REGISTRATION !== 'false',
    enableGuestCheckout: process.env.ENABLE_GUEST_CHECKOUT === 'true',
    enableTwoFactorAuth: process.env.ENABLE_2FA === 'true',
    enableNewsletterSignup: process.env.ENABLE_NEWSLETTER === 'true',
    enableLiveChat: process.env.ENABLE_LIVE_CHAT === 'true',
    enableMaintenanceMode: process.env.MAINTENANCE_MODE === 'true',
    enableBetaFeatures: process.env.ENABLE_BETA_FEATURES === 'true',
  },

  // Third-party integrations
  integrations: {
    googleAnalytics: {
      enabled: process.env.GA_ENABLED === 'true',
      trackingId: process.env.GA_TRACKING_ID,
    },
    googleTagManager: {
      enabled: process.env.GTM_ENABLED === 'true',
      containerId: process.env.GTM_CONTAINER_ID,
    },
    hotjar: {
      enabled: process.env.HOTJAR_ENABLED === 'true',
      siteId: process.env.HOTJAR_SITE_ID,
    },
    intercom: {
      enabled: process.env.INTERCOM_ENABLED === 'true',
      appId: process.env.INTERCOM_APP_ID,
    },
    slack: {
      webhookUrl: process.env.SLACK_WEBHOOK_URL,
      channel: process.env.SLACK_CHANNEL || '#alerts',
    },
  },

  // Performance configuration
  performance: {
    enableCompression: process.env.ENABLE_COMPRESSION !== 'false',
    enableMinification: process.env.ENABLE_MINIFICATION !== 'false',
    enableTreeShaking: process.env.ENABLE_TREE_SHAKING !== 'false',
    enableCodeSplitting: process.env.ENABLE_CODE_SPLITTING !== 'false',
    bundleAnalyzer: process.env.ANALYZE_BUNDLE === 'true',
    enableServiceWorker: process.env.ENABLE_SERVICE_WORKER === 'true',
  },

  // SEO configuration
  seo: {
    siteName: 'NuwairSystems - Professional Web Development Services',
    siteDescription: 'Expert web development, custom software solutions, and digital transformation services. Build modern, scalable applications with our experienced team.',
    siteUrl: process.env.NEXT_PUBLIC_SITE_URL || 'https://nuwairsystems.com',
    defaultImage: '/images/og-default.jpg',
    twitterHandle: '@nuwairsystems',
    facebookAppId: process.env.FACEBOOK_APP_ID,
  },
}

// Environment variable validation
const requiredEnvVars = [
  'DATABASE_URL',
  'NEXTAUTH_SECRET',
  'NEXTAUTH_URL',
  'STRIPE_SECRET_KEY',
  'NEXT_PUBLIC_STRIPE_PUBLISHABLE_KEY',
]

export function validateEnvironment(): { valid: boolean; missing: string[] } {
  const missing = requiredEnvVars.filter(
    varName => !process.env[varName] || process.env[varName]?.trim() === ''
  )

  return {
    valid: missing.length === 0,
    missing,
  }
}

// Configuration validation
export function validateConfiguration(): { valid: boolean; errors: string[] } {
  const errors: string[] = []
  const envValidation = validateEnvironment()

  if (!envValidation.valid) {
    errors.push(`Missing required environment variables: ${envValidation.missing.join(', ')}`)
  }

  // Validate database URL format
  if (productionConfig.database.url && !productionConfig.database.url.startsWith('postgresql://')) {
    errors.push('DATABASE_URL must be a valid PostgreSQL connection string')
  }

  // Validate Stripe keys
  if (productionConfig.payment.stripe.secretKey && !productionConfig.payment.stripe.secretKey.startsWith('sk_')) {
    errors.push('STRIPE_SECRET_KEY must be a valid Stripe secret key')
  }

  if (productionConfig.payment.stripe.publishableKey && !productionConfig.payment.stripe.publishableKey.startsWith('pk_')) {
    errors.push('NEXT_PUBLIC_STRIPE_PUBLISHABLE_KEY must be a valid Stripe publishable key')
  }

  // Validate NextAuth configuration
  if (productionConfig.auth.secret && productionConfig.auth.secret.length < 32) {
    errors.push('NEXTAUTH_SECRET must be at least 32 characters long')
  }

  return {
    valid: errors.length === 0,
    errors,
  }
}

// Get configuration for specific environment
export function getConfig() {
  const validation = validateConfiguration()
  
  if (!validation.valid && process.env.NODE_ENV === 'production') {
    throw new Error(`Configuration validation failed: ${validation.errors.join(', ')}`)
  }

  return productionConfig
}

// Export individual configurations for easier imports
export const {
  server: serverConfig,
  database: databaseConfig,
  auth: authConfig,
  payment: paymentConfig,
  email: emailConfig,
  upload: uploadConfig,
  cache: cacheConfig,
  monitoring: monitoringConfig,
  rateLimit: rateLimitConfig,
  cors: corsConfig,
  cdn: cdnConfig,
  features: featureFlags,
  integrations: integrationsConfig,
  performance: performanceConfig,
  seo: seoConfig,
} = productionConfig