import { NextRequest } from 'next/server'

// Advanced logging system for production monitoring
export class Logger {
  private static readonly LOG_LEVELS = {
    debug: 0,
    info: 1,
    warn: 2,
    error: 3,
  } as const

  private static logLevel = process.env.NODE_ENV === 'production' ? 2 : 0
  private static logs: LogEntry[] = []
  private static readonly MAX_LOGS = 1000

  static debug(message: string, meta?: Record<string, any>): void {
    this.log('debug', message, meta)
  }

  static info(message: string, meta?: Record<string, any>): void {
    this.log('info', message, meta)
  }

  static warn(message: string, meta?: Record<string, any>): void {
    this.log('warn', message, meta)
  }

  static error(message: string, error?: Error, meta?: Record<string, any>): void {
    this.log('error', message, {
      ...meta,
      error: error ? {
        name: error.name,
        message: error.message,
        stack: error.stack,
      } : undefined,
    })
  }

  private static log(level: keyof typeof Logger.LOG_LEVELS, message: string, meta?: Record<string, any>): void {
    if (this.LOG_LEVELS[level] < this.logLevel) {
      return
    }

    const entry: LogEntry = {
      timestamp: new Date().toISOString(),
      level,
      message: this.sanitizeMessage(message),
      meta: meta ? this.sanitizeMeta(meta) : undefined,
      pid: process.pid,
      environment: process.env.NODE_ENV || 'development',
    }

    // Store in memory for monitoring
    this.logs.push(entry)
    if (this.logs.length > this.MAX_LOGS) {
      this.logs.shift()
    }

    // Output to console
    this.outputLog(entry)

    // In production, you might send to external logging service
    if (process.env.NODE_ENV === 'production') {
      this.sendToExternalLogger(entry)
    }
  }

  private static sanitizeMessage(message: string): string {
    // Remove sensitive information from log messages
    const sensitivePatterns = [
      /password[=:]\s*\S+/gi,
      /token[=:]\s*\S+/gi,
      /key[=:]\s*\S+/gi,
      /secret[=:]\s*\S+/gi,
      /\b\d{4}[\s-]?\d{4}[\s-]?\d{4}[\s-]?\d{4}\b/g, // Credit card numbers
      /\b\d{3}[-.]?\d{2}[-.]?\d{4}\b/g, // SSN
    ]

    let sanitized = message
    sensitivePatterns.forEach(pattern => {
      sanitized = sanitized.replace(pattern, '[REDACTED]')
    })

    return sanitized
  }

  private static sanitizeMeta(meta: Record<string, any>): Record<string, any> {
    const sensitiveFields = ['password', 'token', 'secret', 'key', 'creditCard', 'ssn', 'apiKey']
    const sanitized = { ...meta }

    const sanitizeValue = (obj: any): any => {
      if (obj === null || obj === undefined) return obj
      if (typeof obj === 'string') return this.sanitizeMessage(obj)
      if (Array.isArray(obj)) return obj.map(sanitizeValue)
      if (typeof obj === 'object') {
        const result: any = {}
        for (const [key, value] of Object.entries(obj)) {
          if (sensitiveFields.some(field => key.toLowerCase().includes(field.toLowerCase()))) {
            result[key] = '[REDACTED]'
          } else {
            result[key] = sanitizeValue(value)
          }
        }
        return result
      }
      return obj
    }

    return sanitizeValue(sanitized)
  }

  private static outputLog(entry: LogEntry): void {
    const { timestamp, level, message, meta } = entry
    const metaStr = meta ? ` ${JSON.stringify(meta)}` : ''
    
    const logLine = `[${timestamp}] ${level.toUpperCase()}: ${message}${metaStr}`

    switch (level) {
      case 'debug':
        console.debug(logLine)
        break
      case 'info':
        console.info(logLine)
        break
      case 'warn':
        console.warn(logLine)
        break
      case 'error':
        console.error(logLine)
        break
    }
  }

  private static sendToExternalLogger(entry: LogEntry): void {
    // In a real application, you would send logs to services like:
    // - Datadog
    // - New Relic
    // - CloudWatch
    // - Splunk
    // - ELK Stack
    
    // For now, just a placeholder
    if (process.env.EXTERNAL_LOGGING_ENDPOINT) {
      fetch(process.env.EXTERNAL_LOGGING_ENDPOINT, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(entry),
      }).catch(() => {
        // Silently fail to avoid logging loops
      })
    }
  }

  static getLogs(filter?: { level?: string; since?: string; limit?: number }): LogEntry[] {
    let filtered = [...this.logs]

    if (filter?.level) {
      filtered = filtered.filter(log => log.level === filter.level)
    }

    if (filter?.since) {
      const since = new Date(filter.since)
      filtered = filtered.filter(log => new Date(log.timestamp) >= since)
    }

    if (filter?.limit) {
      filtered = filtered.slice(-filter.limit)
    }

    return filtered
  }

  static clearLogs(): void {
    this.logs = []
  }

  static getStats() {
    const stats: Record<string, number> = {}
    this.logs.forEach(log => {
      stats[log.level] = (stats[log.level] || 0) + 1
    })
    return {
      total: this.logs.length,
      byLevel: stats,
    }
  }
}

interface LogEntry {
  timestamp: string
  level: 'debug' | 'info' | 'warn' | 'error'
  message: string
  meta?: Record<string, any>
  pid: number
  environment: string
}

// Request monitoring and analytics
export class RequestMonitor {
  private static requests: RequestMetric[] = []
  private static readonly MAX_REQUESTS = 10000

  static trackRequest(req: NextRequest, startTime: number): () => void {
    const requestId = this.generateRequestId()
    
    Logger.info('Request started', {
      requestId,
      method: req.method,
      url: req.url,
      userAgent: req.headers.get('user-agent'),
      ip: this.getClientIP(req),
    })

    return () => {
      const duration = Date.now() - startTime
      const metric: RequestMetric = {
        id: requestId,
        timestamp: new Date().toISOString(),
        method: req.method,
        url: req.url,
        duration,
        userAgent: req.headers.get('user-agent') || '',
        ip: this.getClientIP(req),
        statusCode: 0, // Will be set by response middleware
      }

      this.requests.push(metric)
      if (this.requests.length > this.MAX_REQUESTS) {
        this.requests.shift()
      }

      Logger.info('Request completed', {
        requestId,
        duration: `${duration}ms`,
        statusCode: metric.statusCode,
      })

      // Alert on slow requests
      if (duration > 5000) {
        Logger.warn('Slow request detected', {
          requestId,
          duration: `${duration}ms`,
          url: req.url,
        })
      }
    }
  }

  static updateResponseStatus(requestId: string, statusCode: number): void {
    const request = this.requests.find(r => r.id === requestId)
    if (request) {
      request.statusCode = statusCode
    }
  }

  private static generateRequestId(): string {
    return Math.random().toString(36).substring(2, 15) + 
           Math.random().toString(36).substring(2, 15)
  }

  private static getClientIP(req: NextRequest): string {
    const forwarded = req.headers.get('x-forwarded-for')
    const realIp = req.headers.get('x-real-ip')
    const cfConnectingIp = req.headers.get('cf-connecting-ip')
    
    return cfConnectingIp || 
           realIp || 
           (forwarded ? forwarded.split(',')[0].trim() : '') ||
           '127.0.0.1'
  }

  static getMetrics(timeRange?: { start: string; end: string }) {
    let filtered = [...this.requests]

    if (timeRange) {
      const start = new Date(timeRange.start)
      const end = new Date(timeRange.end)
      filtered = filtered.filter(r => {
        const timestamp = new Date(r.timestamp)
        return timestamp >= start && timestamp <= end
      })
    }

    const totalRequests = filtered.length
    const avgDuration = filtered.reduce((sum, r) => sum + r.duration, 0) / totalRequests || 0
    const statusCodes = filtered.reduce((acc, r) => {
      acc[r.statusCode] = (acc[r.statusCode] || 0) + 1
      return acc
    }, {} as Record<number, number>)

    const slowRequests = filtered.filter(r => r.duration > 2000).length
    const errorRequests = filtered.filter(r => r.statusCode >= 400).length

    return {
      totalRequests,
      avgDuration: Math.round(avgDuration),
      slowRequests,
      errorRequests,
      errorRate: totalRequests > 0 ? (errorRequests / totalRequests) * 100 : 0,
      statusCodes,
    }
  }

  static getTopEndpoints(limit = 10) {
    const endpoints = this.requests.reduce((acc, r) => {
      const url = new URL(r.url).pathname
      if (!acc[url]) {
        acc[url] = { count: 0, totalDuration: 0, errors: 0 }
      }
      acc[url].count++
      acc[url].totalDuration += r.duration
      if (r.statusCode >= 400) {
        acc[url].errors++
      }
      return acc
    }, {} as Record<string, { count: number; totalDuration: number; errors: number }>)

    return Object.entries(endpoints)
      .map(([url, stats]) => ({
        url,
        count: stats.count,
        avgDuration: Math.round(stats.totalDuration / stats.count),
        errorRate: (stats.errors / stats.count) * 100,
      }))
      .sort((a, b) => b.count - a.count)
      .slice(0, limit)
  }

  static clearMetrics(): void {
    this.requests = []
  }
}

interface RequestMetric {
  id: string
  timestamp: string
  method: string
  url: string
  duration: number
  userAgent: string
  ip: string
  statusCode: number
}

// Error tracking and reporting
export class ErrorTracker {
  private static errors: ErrorReport[] = []
  private static readonly MAX_ERRORS = 1000

  static captureError(error: Error, context?: Record<string, any>): string {
    const errorId = this.generateErrorId()
    
    const report: ErrorReport = {
      id: errorId,
      timestamp: new Date().toISOString(),
      name: error.name,
      message: error.message,
      stack: error.stack || '',
      context: context || {},
      environment: process.env.NODE_ENV || 'development',
      userAgent: context?.userAgent || '',
      url: context?.url || '',
      userId: context?.userId || null,
    }

    this.errors.push(report)
    if (this.errors.length > this.MAX_ERRORS) {
      this.errors.shift()
    }

    Logger.error('Error captured', error, { errorId, context })

    // In production, send to error tracking service
    if (process.env.NODE_ENV === 'production') {
      this.sendToErrorTrackingService(report)
    }

    return errorId
  }

  private static generateErrorId(): string {
    return 'err_' + Math.random().toString(36).substring(2, 15)
  }

  private static sendToErrorTrackingService(report: ErrorReport): void {
    // In a real application, you would send to services like:
    // - Sentry
    // - Rollbar
    // - Bugsnag
    // - DataDog
    
    if (process.env.ERROR_TRACKING_ENDPOINT) {
      fetch(process.env.ERROR_TRACKING_ENDPOINT, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(report),
      }).catch(() => {
        // Silently fail to avoid error loops
      })
    }
  }

  static getErrors(filter?: { since?: string; limit?: number; userId?: string }) {
    let filtered = [...this.errors]

    if (filter?.since) {
      const since = new Date(filter.since)
      filtered = filtered.filter(e => new Date(e.timestamp) >= since)
    }

    if (filter?.userId) {
      filtered = filtered.filter(e => e.userId === filter.userId)
    }

    if (filter?.limit) {
      filtered = filtered.slice(-filter.limit)
    }

    return filtered
  }

  static getErrorStats() {
    const now = new Date()
    const oneHourAgo = new Date(now.getTime() - 60 * 60 * 1000)
    const oneDayAgo = new Date(now.getTime() - 24 * 60 * 60 * 1000)

    const recentErrors = this.errors.filter(e => new Date(e.timestamp) >= oneHourAgo)
    const dailyErrors = this.errors.filter(e => new Date(e.timestamp) >= oneDayAgo)

    const errorsByType = this.errors.reduce((acc, e) => {
      acc[e.name] = (acc[e.name] || 0) + 1
      return acc
    }, {} as Record<string, number>)

    return {
      total: this.errors.length,
      lastHour: recentErrors.length,
      lastDay: dailyErrors.length,
      byType: errorsByType,
    }
  }

  static clearErrors(): void {
    this.errors = []
  }
}

interface ErrorReport {
  id: string
  timestamp: string
  name: string
  message: string
  stack: string
  context: Record<string, any>
  environment: string
  userAgent: string
  url: string
  userId: string | null
}

// Health monitoring
export class HealthMonitor {
  static async checkHealth(): Promise<HealthStatus> {
    const checks = await Promise.allSettled([
      this.checkDatabase(),
      this.checkMemory(),
      this.checkDisk(),
      this.checkExternalServices(),
    ])

    const results = checks.map((check, index) => ({
      name: ['database', 'memory', 'disk', 'external_services'][index],
      status: check.status === 'fulfilled' ? check.value.status : 'unhealthy',
      details: check.status === 'fulfilled' ? check.value.details : { error: (check.reason as Error).message },
    }))

    const overallStatus = results.every(r => r.status === 'healthy') ? 'healthy' : 'unhealthy'

    return {
      status: overallStatus,
      timestamp: new Date().toISOString(),
      checks: results,
      uptime: process.uptime(),
      version: process.env.npm_package_version || '1.0.0',
    }
  }

  private static async checkDatabase(): Promise<HealthCheck> {
    try {
      // In a real app, you would test the database connection
      // const result = await prisma.$queryRaw`SELECT 1`
      return {
        status: 'healthy',
        details: { connection: 'ok', latency: '< 10ms' },
      }
    } catch (error) {
      return {
        status: 'unhealthy',
        details: { error: (error as Error).message },
      }
    }
  }

  private static async checkMemory(): Promise<HealthCheck> {
    const usage = process.memoryUsage()
    const usagePercent = (usage.heapUsed / usage.heapTotal) * 100

    return {
      status: usagePercent < 80 ? 'healthy' : 'unhealthy',
      details: {
        heapUsed: Math.round(usage.heapUsed / 1024 / 1024) + 'MB',
        heapTotal: Math.round(usage.heapTotal / 1024 / 1024) + 'MB',
        usagePercent: Math.round(usagePercent) + '%',
      },
    }
  }

  private static async checkDisk(): Promise<HealthCheck> {
    // In a real app, you would check disk space
    return {
      status: 'healthy',
      details: { available: '> 1GB' },
    }
  }

  private static async checkExternalServices(): Promise<HealthCheck> {
    try {
      // Check Stripe API
      if (process.env.STRIPE_SECRET_KEY) {
        // In a real app, you would make a test API call
        // await stripe.balance.retrieve()
      }

      return {
        status: 'healthy',
        details: { stripe: 'ok' },
      }
    } catch (error) {
      return {
        status: 'unhealthy',
        details: { error: (error as Error).message },
      }
    }
  }
}

interface HealthStatus {
  status: 'healthy' | 'unhealthy'
  timestamp: string
  checks: Array<{
    name: string
    status: 'healthy' | 'unhealthy'
    details: Record<string, any>
  }>
  uptime: number
  version: string
}

interface HealthCheck {
  status: 'healthy' | 'unhealthy'
  details: Record<string, any>
}