import { NextResponse } from 'next/server'

// Cache utilities for performance optimization
export class CacheManager {
  private static cache = new Map<string, { data: any; expiry: number }>()
  private static readonly DEFAULT_TTL = 5 * 60 * 1000 // 5 minutes

  static set(key: string, data: any, ttl: number = this.DEFAULT_TTL): void {
    const expiry = Date.now() + ttl
    this.cache.set(key, { data, expiry })
    
    // Auto-cleanup expired entries
    setTimeout(() => {
      this.cleanup()
    }, ttl)
  }

  static get<T>(key: string): T | null {
    const item = this.cache.get(key)
    
    if (!item) {
      return null
    }

    if (Date.now() > item.expiry) {
      this.cache.delete(key)
      return null
    }

    return item.data as T
  }

  static delete(key: string): boolean {
    return this.cache.delete(key)
  }

  static clear(): void {
    this.cache.clear()
  }

  static cleanup(): void {
    const now = Date.now()
    for (const [key, item] of this.cache.entries()) {
      if (now > item.expiry) {
        this.cache.delete(key)
      }
    }
  }

  static getStats() {
    return {
      size: this.cache.size,
      keys: Array.from(this.cache.keys()),
    }
  }
}

// Performance monitoring utilities
export class PerformanceMonitor {
  private static metrics = new Map<string, number[]>()

  static startTimer(label: string): () => number {
    const start = performance.now()
    
    return () => {
      const duration = performance.now() - start
      this.recordMetric(label, duration)
      return duration
    }
  }

  static recordMetric(label: string, value: number): void {
    if (!this.metrics.has(label)) {
      this.metrics.set(label, [])
    }
    
    const values = this.metrics.get(label)!
    values.push(value)
    
    // Keep only last 100 measurements
    if (values.length > 100) {
      values.shift()
    }
  }

  static getMetrics(label: string) {
    const values = this.metrics.get(label) || []
    
    if (values.length === 0) {
      return null
    }

    const avg = values.reduce((a, b) => a + b, 0) / values.length
    const min = Math.min(...values)
    const max = Math.max(...values)
    const p95 = values.sort((a, b) => a - b)[Math.floor(values.length * 0.95)]

    return { avg, min, max, p95, count: values.length }
  }

  static getAllMetrics() {
    const result: Record<string, any> = {}
    
    for (const [label] of this.metrics) {
      result[label] = this.getMetrics(label)
    }
    
    return result
  }

  static clearMetrics(): void {
    this.metrics.clear()
  }
}

// Response compression and optimization
export async function optimizeResponse(response: NextResponse, data?: any): Promise<NextResponse> {
  // Set performance headers
  response.headers.set('Cache-Control', 'public, s-maxage=3600, stale-while-revalidate=86400')
  response.headers.set('X-Content-Type-Options', 'nosniff')
  
  // Enable compression for larger responses
  if (data && JSON.stringify(data).length > 1024) {
    response.headers.set('Content-Encoding', 'gzip')
  }

  // Set ETag for caching
  if (data) {
    const etag = await generateETag(data)
    response.headers.set('ETag', etag)
  }

  return response
}

async function generateETag(data: any): Promise<string> {
  const hash = Array.from(
    new Uint8Array(
      await crypto.subtle.digest('SHA-256', new TextEncoder().encode(JSON.stringify(data)))
    )
  ).map(b => b.toString(16).padStart(2, '0')).join('')
  
  return `"${hash.substring(0, 16)}"`
}

// Database query optimization
export class QueryOptimizer {
  private static slowQueries = new Map<string, { count: number; totalTime: number }>()

  static async trackQuery<T>(
    queryName: string,
    queryFn: () => Promise<T>
  ): Promise<T> {
    const endTimer = PerformanceMonitor.startTimer(`db_query_${queryName}`)
    
    try {
      const result = await queryFn()
      const duration = endTimer()
      
      // Track slow queries (> 1 second)
      if (duration > 1000) {
        this.recordSlowQuery(queryName, duration)
      }
      
      return result
    } catch (error) {
      endTimer()
      throw error
    }
  }

  private static recordSlowQuery(queryName: string, duration: number): void {
    const existing = this.slowQueries.get(queryName) || { count: 0, totalTime: 0 }
    this.slowQueries.set(queryName, {
      count: existing.count + 1,
      totalTime: existing.totalTime + duration,
    })
  }

  static getSlowQueries() {
    const result: Record<string, any> = {}
    
    for (const [queryName, stats] of this.slowQueries) {
      result[queryName] = {
        ...stats,
        avgTime: stats.totalTime / stats.count,
      }
    }
    
    return result
  }

  static clearSlowQueries(): void {
    this.slowQueries.clear()
  }
}

// Image optimization utilities
export class ImageOptimizer {
  static readonly SUPPORTED_FORMATS = ['jpeg', 'jpg', 'png', 'webp', 'avif']
  static readonly MAX_WIDTH = 1920
  static readonly MAX_HEIGHT = 1080
  static readonly QUALITY = 85

  static isImageFile(filename: string): boolean {
    const ext = filename.split('.').pop()?.toLowerCase()
    return ext ? this.SUPPORTED_FORMATS.includes(ext) : false
  }

  static generateImageSizes(originalWidth: number, originalHeight: number) {
    const sizes = [640, 768, 1024, 1280, 1920]
    const aspectRatio = originalHeight / originalWidth
    
    return sizes
      .filter(width => width <= originalWidth)
      .map(width => ({
        width,
        height: Math.round(width * aspectRatio),
      }))
  }

  static getOptimizedImageUrl(
    src: string,
    width?: number,
    height?: number,
    quality?: number
  ): string {
    const url = new URL(src, 'https://example.com')
    
    if (width) url.searchParams.set('w', width.toString())
    if (height) url.searchParams.set('h', height.toString())
    if (quality) url.searchParams.set('q', quality.toString())
    
    return url.pathname + url.search
  }
}

// Bundle analysis and optimization
export class BundleAnalyzer {
  static analyzePageBundle(pageName: string) {
    // This would integrate with webpack-bundle-analyzer in a real implementation
    return {
      totalSize: 0,
      gzippedSize: 0,
      chunks: [],
      unusedExports: [],
      duplicateModules: [],
    }
  }

  static getOptimizationSuggestions() {
    return [
      'Enable code splitting for large components',
      'Use dynamic imports for non-critical features',
      'Implement tree shaking for unused exports',
      'Optimize images and use next/image',
      'Enable gzip compression',
      'Implement service worker for caching',
      'Use CDN for static assets',
      'Minimize CSS and JavaScript bundles',
    ]
  }
}

// Memory management utilities
export class MemoryManager {
  private static readonly MAX_MEMORY_USAGE = 0.8 // 80% of available memory

  static checkMemoryUsage(): { usage: number; isHigh: boolean } {
    if (typeof window !== 'undefined' && 'memory' in performance) {
      const memory = (performance as any).memory
      const usage = memory.usedJSHeapSize / memory.jsHeapSizeLimit
      
      return {
        usage,
        isHigh: usage > this.MAX_MEMORY_USAGE,
      }
    }
    
    return { usage: 0, isHigh: false }
  }

  static forceGarbageCollection(): void {
    if (typeof window !== 'undefined' && 'gc' in window) {
      (window as any).gc()
    }
  }

  static clearCaches(): void {
    CacheManager.clear()
    PerformanceMonitor.clearMetrics()
    QueryOptimizer.clearSlowQueries()
  }
}

// Performance configuration
export const performanceConfig = {
  cache: {
    ttl: {
      static: 24 * 60 * 60 * 1000, // 24 hours
      api: 5 * 60 * 1000, // 5 minutes
      user: 15 * 60 * 1000, // 15 minutes
      search: 10 * 60 * 1000, // 10 minutes
    },
    maxSize: 100, // Maximum number of cached items
  },
  
  database: {
    connectionPool: {
      min: 2,
      max: 10,
      idle: 10000,
    },
    queryTimeout: 15000,
    maxQueryComplexity: 1000,
  },
  
  images: {
    quality: 85,
    formats: ['webp', 'avif', 'jpeg'],
    sizes: [640, 768, 1024, 1280, 1920],
    placeholder: 'blur',
  },
  
  bundleSize: {
    maxInitialJS: 244 * 1024, // 244KB
    maxPageJS: 128 * 1024, // 128KB
    maxCSS: 32 * 1024, // 32KB
  },
}