import { NextRequest } from 'next/server'

interface RateLimitEntry {
  count: number
  resetTime: number
}

interface TokenBucketEntry {
  tokens: number
  lastRefill: number
}

// Separate stores for different rate limiting strategies
const rateLimitStore: { [key: string]: RateLimitEntry } = {}
const tokenBucketStore: { [key: string]: TokenBucketEntry } = {}

// Clear all rate limit entries (useful for development/testing)
export function clearRateLimit() {
  Object.keys(rateLimitStore).forEach(key => delete rateLimitStore[key])
  Object.keys(tokenBucketStore).forEach(key => delete tokenBucketStore[key])
  console.log('Rate limit cache cleared')
}

// Clean up expired entries every 5 minutes
setInterval(() => {
  const now = Date.now()
  Object.keys(rateLimitStore).forEach(key => {
    if (rateLimitStore[key].resetTime < now) {
      delete rateLimitStore[key]
    }
  })
}, 5 * 60 * 1000)

export async function rateLimit(
  identifier: string,
  limit: number,
  windowMs: string
): Promise<{ success: boolean; limit: number; remaining: number; resetTime: number }> {
  const now = Date.now()
  const windowMsNumber = parseTimeWindow(windowMs)
  const key = `rate_limit:${identifier}`
  
  // Get or create rate limit entry
  let entry = rateLimitStore[key]
  
  if (!entry || entry.resetTime < now) {
    // Create new entry or reset expired one
    entry = {
      count: 0,
      resetTime: now + windowMsNumber
    }
    rateLimitStore[key] = entry
  }
  
  // Increment count
  entry.count++
  
  const success = entry.count <= limit
  const remaining = Math.max(0, limit - entry.count)
  
  return {
    success,
    limit,
    remaining,
    resetTime: entry.resetTime
  }
}

function parseTimeWindow(window: string): number {
  const match = window.match(/^(\d+)([smhd])$/)
  if (!match) {
    throw new Error('Invalid time window format. Use format like "15m", "1h", "1d"')
  }
  
  const value = parseInt(match[1])
  const unit = match[2]
  
  switch (unit) {
    case 's': return value * 1000
    case 'm': return value * 60 * 1000
    case 'h': return value * 60 * 60 * 1000
    case 'd': return value * 24 * 60 * 60 * 1000
    default: throw new Error('Invalid time unit')
  }
}

export function getClientIP(request: NextRequest): string {
  // Try to get IP from various headers
  const forwarded = request.headers.get('x-forwarded-for')
  const realIp = request.headers.get('x-real-ip')
  const cfConnectingIp = request.headers.get('cf-connecting-ip')
  
  if (cfConnectingIp) return cfConnectingIp
  if (realIp) return realIp
  if (forwarded) return forwarded.split(',')[0].trim()
  
  return 'unknown'
}

// Advanced rate limiting with different strategies
export class RateLimiter {
  async slidingWindow(
    identifier: string,
    limit: number,
    windowMs: number
  ): Promise<{ success: boolean; retryAfter?: number }> {
    const now = Date.now()
    const key = `sliding:${identifier}`
    
    let entry = rateLimitStore[key] || { count: 0, resetTime: now + windowMs }
    
    // Slide the window
    if (now >= entry.resetTime) {
      entry = { count: 0, resetTime: now + windowMs }
    }
    
    entry.count++
    rateLimitStore[key] = entry
    
    if (entry.count > limit) {
      return {
        success: false,
        retryAfter: Math.ceil((entry.resetTime - now) / 1000)
      }
    }
    
    return { success: true }
  }
  
  async tokenBucket(
    identifier: string,
    capacity: number,
    refillRate: number, // tokens per second
    tokensRequested: number = 1
  ): Promise<{ success: boolean; tokensRemaining: number }> {
    const now = Date.now()
    const key = `bucket:${identifier}`
    
    let bucket = tokenBucketStore[key] || {
      tokens: capacity,
      lastRefill: now
    }
    
    // Refill tokens based on time passed
    const timePassed = (now - bucket.lastRefill) / 1000
    const tokensToAdd = Math.floor(timePassed * refillRate)
    
    bucket.tokens = Math.min(capacity, bucket.tokens + tokensToAdd)
    bucket.lastRefill = now
    
    if (bucket.tokens >= tokensRequested) {
      bucket.tokens -= tokensRequested
      tokenBucketStore[key] = bucket
      return { success: true, tokensRemaining: bucket.tokens }
    }
    
    tokenBucketStore[key] = bucket
    return { success: false, tokensRemaining: bucket.tokens }
  }
}