// Security configuration for production deployment
export const securityConfig = {
  // Password requirements
  password: {
    minLength: 8,
    requireUppercase: true,
    requireLowercase: true,
    requireNumbers: true,
    requireSpecialChars: true,
    maxAttempts: 5,
    lockoutDuration: 15 * 60 * 1000, // 15 minutes
  },

  // Session security
  session: {
    maxAge: 7 * 24 * 60 * 60, // 7 days
    updateAge: 24 * 60 * 60, // 1 day
    secure: process.env.NODE_ENV === 'production',
    httpOnly: true,
    sameSite: 'strict' as const,
  },

  // File upload restrictions
  upload: {
    maxSize: 10 * 1024 * 1024, // 10MB
    allowedTypes: [
      'image/jpeg',
      'image/png',
      'image/webp',
      'image/svg+xml',
      'application/pdf',
      'text/plain',
      'application/msword',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
    ],
    maxFiles: 5,
    scanForViruses: true,
  },

  // API rate limiting
  rateLimit: {
    api: { requests: 100, window: '15m' },
    auth: { requests: 5, window: '15m' },
    payment: { requests: 10, window: '15m' },
    upload: { requests: 20, window: '15m' },
    forgotten: { requests: 3, window: '1h' },
  },

  // Input validation
  validation: {
    maxStringLength: 10000,
    maxArrayLength: 1000,
    maxDepth: 10,
    sanitizeHtml: true,
    allowedTags: ['b', 'i', 'em', 'strong', 'p', 'br', 'ul', 'ol', 'li'],
  },

  // CORS configuration
  cors: {
    origin: process.env.NODE_ENV === 'production' 
      ? ['https://nuwairsystems.com', 'https://www.nuwairsystems.com']
      : ['http://localhost:3000'],
    credentials: true,
    maxAge: 86400, // 24 hours
  },

  // Security headers
  headers: {
    'X-Content-Type-Options': 'nosniff',
    'X-Frame-Options': 'DENY',
    'X-XSS-Protection': '1; mode=block',
    'Strict-Transport-Security': 'max-age=31536000; includeSubDomains; preload',
    'Referrer-Policy': 'strict-origin-when-cross-origin',
    'Permissions-Policy': [
      'camera=()',
      'microphone=()',
      'geolocation=()',
      'interest-cohort=()',
    ].join(', '),
  },

  // Content Security Policy
  csp: {
    'default-src': ["'self'"],
    'script-src': [
      "'self'",
      "'unsafe-eval'",
      "'unsafe-inline'",
      'https://js.stripe.com',
      'https://challenges.cloudflare.com',
    ],
    'style-src': [
      "'self'",
      "'unsafe-inline'",
      'https://fonts.googleapis.com',
    ],
    'font-src': [
      "'self'",
      'https://fonts.gstatic.com',
    ],
    'img-src': [
      "'self'",
      'data:',
      'https:',
    ],
    'connect-src': [
      "'self'",
      'https://api.stripe.com',
      'https://vitals.vercel-insights.com',
      'https://challenges.cloudflare.com',
    ],
    'frame-src': [
      'https://js.stripe.com',
      'https://hooks.stripe.com',
      'https://challenges.cloudflare.com',
    ],
    'object-src': ["'none'"],
    'base-uri': ["'self'"],
    'form-action': ["'self'"],
    'frame-ancestors': ["'none'"],
    'upgrade-insecure-requests': [],
  },

  // Database security
  database: {
    connectionTimeout: 30000,
    queryTimeout: 15000,
    maxConnections: 10,
    encryptSensitiveFields: true,
    auditLogs: true,
  },

  // Logging and monitoring
  logging: {
    level: process.env.NODE_ENV === 'production' ? 'warn' : 'debug',
    includeStack: process.env.NODE_ENV !== 'production',
    maxLogSize: 10 * 1024 * 1024, // 10MB
    retentionDays: 30,
    sensitiveFields: [
      'password',
      'token',
      'secret',
      'key',
      'creditCard',
      'ssn',
      'apiKey',
    ],
  },

  // API security
  api: {
    requireAuth: true,
    validateInput: true,
    sanitizeOutput: true,
    logRequests: true,
    maxRequestSize: 10 * 1024 * 1024, // 10MB
    timeout: 30000, // 30 seconds
  },

  // Environment validation
  requiredEnvVars: [
    'DATABASE_URL',
    'NEXTAUTH_SECRET',
    'NEXTAUTH_URL',
    'STRIPE_SECRET_KEY',
    'NEXT_PUBLIC_STRIPE_PUBLISHABLE_KEY',
  ],

  // Production-only features
  production: {
    enableCSRFProtection: true,
    enableHSTS: true,
    enableContentTypeValidation: true,
    enableRequestSizeLimit: true,
    enableSlowLorisProtection: true,
    enableBruteForceProtection: true,
    enableDDoSProtection: true,
  },
}

// Utility functions for security
export const sanitizeInput = (input: string): string => {
  return input
    .replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '')
    .replace(/javascript:/gi, '')
    .replace(/on\w+\s*=/gi, '')
    .trim()
}

export const validateEmail = (email: string): boolean => {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/
  return emailRegex.test(email) && email.length <= 254
}

export const validatePassword = (password: string): { valid: boolean; errors: string[] } => {
  const errors: string[] = []
  const config = securityConfig.password

  if (password.length < config.minLength) {
    errors.push(`Password must be at least ${config.minLength} characters long`)
  }

  if (config.requireUppercase && !/[A-Z]/.test(password)) {
    errors.push('Password must contain at least one uppercase letter')
  }

  if (config.requireLowercase && !/[a-z]/.test(password)) {
    errors.push('Password must contain at least one lowercase letter')
  }

  if (config.requireNumbers && !/\d/.test(password)) {
    errors.push('Password must contain at least one number')
  }

  if (config.requireSpecialChars && !/[!@#$%^&*(),.?":{}|<>]/.test(password)) {
    errors.push('Password must contain at least one special character')
  }

  return {
    valid: errors.length === 0,
    errors,
  }
}

export const generateCSRFToken = (): string => {
  return Array.from(crypto.getRandomValues(new Uint8Array(32)))
    .map(b => b.toString(16).padStart(2, '0'))
    .join('')
}

export const isValidFileType = (mimeType: string): boolean => {
  return securityConfig.upload.allowedTypes.includes(mimeType)
}

export const logSecurityEvent = (event: string, details: Record<string, any>) => {
  if (process.env.NODE_ENV === 'production') {
    console.warn(`SECURITY EVENT: ${event}`, {
      timestamp: new Date().toISOString(),
      ...details,
    })
  }
}