import Stripe from 'stripe'
import { loadStripe, Stripe as StripeJS } from '@stripe/stripe-js'

// Server-side Stripe instance with error handling
const stripe = process.env.STRIPE_SECRET_KEY 
  ? new Stripe(process.env.STRIPE_SECRET_KEY, {
      apiVersion: '2025-09-30.clover',
      typescript: true,
    })
  : null

// Client-side Stripe instance with error handling
let stripePromise: Promise<StripeJS | null>
const getStripe = () => {
  if (!stripePromise) {
    const publishableKey = process.env.NEXT_PUBLIC_STRIPE_PUBLISHABLE_KEY
    if (!publishableKey) {
      console.warn('Stripe publishable key not found')
      stripePromise = Promise.resolve(null)
    } else {
      stripePromise = loadStripe(publishableKey)
    }
  }
  return stripePromise
}

// Payment helper functions
export const createPaymentIntent = async (
  amount: number,
  currency: string = 'usd',
  metadata?: Record<string, string>
) => {
  try {
    if (!stripe) {
      throw new Error('Stripe is not configured')
    }
    
    const paymentIntent = await stripe.paymentIntents.create({
      amount: Math.round(amount * 100), // Convert to cents
      currency,
      automatic_payment_methods: {
        enabled: true,
      },
      metadata,
    })

    return {
      success: true,
      clientSecret: paymentIntent.client_secret,
      paymentIntentId: paymentIntent.id,
    }
  } catch (error) {
    console.error('Error creating payment intent:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
    }
  }
}

export const retrievePaymentIntent = async (paymentIntentId: string) => {
  try {
    if (!stripe) {
      throw new Error('Stripe is not configured')
    }
    
    const paymentIntent = await stripe.paymentIntents.retrieve(paymentIntentId)
    return {
      success: true,
      paymentIntent,
    }
  } catch (error) {
    console.error('Error retrieving payment intent:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
    }
  }
}

export const createInvoice = async (
  customerId: string,
  items: Array<{ description: string; amount: number; quantity?: number }>,
  options?: {
    dueDate?: number // Days from now
    description?: string
    metadata?: Record<string, string>
  }
) => {
  try {
    if (!stripe) {
      throw new Error('Stripe is not configured')
    }
    
    // Create invoice items
    for (const item of items) {
      await stripe.invoiceItems.create({
        customer: customerId,
        amount: Math.round(item.amount * 100),
        currency: 'usd',
        description: item.description,
        quantity: item.quantity || 1,
      })
    }

    // Create the invoice
    const invoice = await stripe.invoices.create({
      customer: customerId,
      collection_method: 'send_invoice',
      days_until_due: options?.dueDate || 30,
      description: options?.description,
      metadata: options?.metadata,
    })

    // Finalize the invoice
    const finalizedInvoice = await stripe.invoices.finalizeInvoice(invoice.id)

    return {
      success: true,
      invoice: finalizedInvoice,
    }
  } catch (error) {
    console.error('Error creating invoice:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
    }
  }
}

export const sendInvoice = async (invoiceId: string) => {
  try {
    if (!stripe) {
      throw new Error('Stripe is not configured')
    }
    
    const invoice = await stripe.invoices.sendInvoice(invoiceId)
    return {
      success: true,
      invoice,
    }
  } catch (error) {
    console.error('Error sending invoice:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
    }
  }
}

export const createCustomer = async (
  email: string,
  name?: string,
  metadata?: Record<string, string>
) => {
  try {
    if (!stripe) {
      throw new Error('Stripe is not configured')
    }
    
    const customer = await stripe.customers.create({
      email,
      name,
      metadata,
    })

    return {
      success: true,
      customer,
    }
  } catch (error) {
    console.error('Error creating customer:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
    }
  }
}

export const refundPayment = async (
  paymentIntentId: string,
  amount?: number,
  reason?: 'duplicate' | 'fraudulent' | 'requested_by_customer'
) => {
  try {
    if (!stripe) {
      throw new Error('Stripe is not configured')
    }
    
    const refund = await stripe.refunds.create({
      payment_intent: paymentIntentId,
      amount: amount ? Math.round(amount * 100) : undefined,
      reason,
    })

    return {
      success: true,
      refund,
    }
  } catch (error) {
    console.error('Error processing refund:', error)
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Unknown error',
    }
  }
}

export { stripe, getStripe }
export default stripe