import { z } from 'zod'

export const signInSchema = z.object({
  email: z.string().email('Invalid email address'),
  password: z.string().min(6, 'Password must be at least 6 characters'),
})

export const signUpSchema = z.object({
  name: z.string().min(2, 'Name must be at least 2 characters'),
  email: z.string().email('Invalid email address'),
  password: z.string().min(6, 'Password must be at least 6 characters'),
  confirmPassword: z.string(),
}).refine((data) => data.password === data.confirmPassword, {
  message: 'Passwords do not match',
  path: ['confirmPassword'],
})

export const contactSchema = z.object({
  name: z.string().min(2, 'Name must be at least 2 characters'),
  email: z.string().email('Invalid email address'),
  subject: z.string().min(5, 'Subject must be at least 5 characters'),
  message: z.string().min(10, 'Message must be at least 10 characters'),
  phone: z.string().optional(),
  company: z.string().optional(),
})

export const checkoutSchema = z.object({
  email: z.string().email('Invalid email address'),
  name: z.string().min(2, 'Name must be at least 2 characters'),
  company: z.string().optional(),
  phone: z.string().min(10, 'Phone number must be at least 10 characters'),
  address: z.string().min(5, 'Address must be at least 5 characters'),
  city: z.string().min(2, 'City must be at least 2 characters'),
  country: z.string().min(2, 'Country must be selected'),
  vatNumber: z.string().optional(),
  notes: z.string().optional(),
})

export const serviceSchema = z.object({
  name: z.string().min(2, 'Service name must be at least 2 characters'),
  slug: z.string().min(2, 'Slug must be at least 2 characters'),
  category: z.enum(['DOMAINS', 'HOSTING', 'MARKETING', 'DEVELOPMENT']),
  description: z.string().min(10, 'Description must be at least 10 characters'),
  shortDescription: z.string().min(5, 'Short description must be at least 5 characters'),
  price: z.number().min(0, 'Price must be non-negative'),
  pricingModel: z.enum(['ONE_TIME', 'MONTHLY', 'YEARLY']),
  features: z.array(z.string()).min(1, 'At least one feature is required'),
  popular: z.boolean().default(false),
  active: z.boolean().default(true),
  image: z.string().optional(),
  configurationOptions: z.record(z.string(), z.any()).optional(),
})

export const ticketSchema = z.object({
  subject: z.string().min(5, 'Subject must be at least 5 characters'),
  description: z.string().min(10, 'Description must be at least 10 characters'),
  priority: z.enum(['LOW', 'MEDIUM', 'HIGH', 'URGENT']).default('MEDIUM'),
})

export const ticketReplySchema = z.object({
  content: z.string().min(5, 'Reply must be at least 5 characters'),
})

export type SignInInput = z.infer<typeof signInSchema>
export type SignUpInput = z.infer<typeof signUpSchema>
export type ContactInput = z.infer<typeof contactSchema>
export type CheckoutInput = z.infer<typeof checkoutSchema>
export type ServiceInput = z.infer<typeof serviceSchema>
export type TicketInput = z.infer<typeof ticketSchema>
export type TicketReplyInput = z.infer<typeof ticketReplySchema>