import { NextRequest, NextResponse } from 'next/server'
import { rateLimit, getClientIP } from '@/lib/rateLimit'
import { getToken } from 'next-auth/jwt'

// Security headers configuration
const securityHeaders = {
  // Prevents XSS attacks
  'X-Content-Type-Options': 'nosniff',
  'X-Frame-Options': 'DENY',
  'X-XSS-Protection': '1; mode=block',
  
  // HTTPS enforcement
  'Strict-Transport-Security': 'max-age=31536000; includeSubDomains; preload',
  
  // Content Security Policy
  'Content-Security-Policy': [
    "default-src 'self'",
    "script-src 'self' 'unsafe-eval' 'unsafe-inline' https://js.stripe.com",
    "style-src 'self' 'unsafe-inline' https://fonts.googleapis.com",
    "font-src 'self' https://fonts.gstatic.com",
    "img-src 'self' data: https:",
    "connect-src 'self' https://api.stripe.com https://vitals.vercel-insights.com",
    "frame-src https://js.stripe.com https://hooks.stripe.com",
    "object-src 'none'",
    "base-uri 'self'",
    "form-action 'self'",
    "frame-ancestors 'none'",
    "upgrade-insecure-requests",
  ].join('; '),
  
  // Referrer policy
  'Referrer-Policy': 'strict-origin-when-cross-origin',
  
  // Permissions policy
  'Permissions-Policy': [
    'camera=()',
    'microphone=()',
    'geolocation=()',
    'interest-cohort=()',
  ].join(', '),
}

// Rate limiting configuration
const rateLimitConfig = {
  api: { requests: 100, window: '15m' },
  auth: { requests: 50, window: '15m' }, // Increased from 5 to 50 for testing
  payment: { requests: 10, window: '15m' },
  upload: { requests: 20, window: '15m' },
}

// Protected admin routes
const adminRoutes = ['/admin', '/api/admin']
const authRoutes = ['/dashboard', '/api/orders', '/api/payments', '/api/invoices']
const publicApiRoutes = ['/api/auth', '/api/domains/check']

export async function middleware(request: NextRequest) {
  // Multiple ways to detect development environment
  const hostname = request.nextUrl.hostname
  const isDevelopment = process.env.NODE_ENV === 'development' || 
                       process.env.NODE_ENV !== 'production' ||
                       hostname === 'localhost' || 
                       hostname === '127.0.0.1' ||
                       request.nextUrl.port === '3000'
  
  // Completely bypass middleware for development
  if (isDevelopment) {
    console.log('Development environment detected - bypassing all middleware')
    const response = NextResponse.next()
    // Add only essential headers
    response.headers.set('X-Frame-Options', 'SAMEORIGIN')
    response.headers.set('X-Content-Type-Options', 'nosniff')
    return response
  }

  const { pathname } = request.nextUrl
  const response = NextResponse.next()

  // Apply security headers to all responses
  Object.entries(securityHeaders).forEach(([key, value]) => {
    response.headers.set(key, value)
  })

  // Apply rate limiting only in production
  const rateLimitResult = await applyRateLimit(request, pathname)
  if (!rateLimitResult.success) {
    return new NextResponse(
      JSON.stringify({
        error: 'Too many requests',
        message: 'Rate limit exceeded. Please try again later.',
      }),
      {
        status: 429,
        headers: {
          'Content-Type': 'application/json',
          'Retry-After': '900', // 15 minutes
          ...Object.fromEntries(Object.entries(securityHeaders)),
        },
      }
    )
  }

  // Admin route protection
  if (adminRoutes.some(route => pathname.startsWith(route))) {
    const token = await getToken({ 
      req: request, 
      secret: process.env.NEXTAUTH_SECRET 
    })
    
    if (!token || token.role !== 'ADMIN') {
      const url = request.nextUrl.clone()
      url.pathname = '/auth/signin'
      url.searchParams.set('callbackUrl', pathname)
      return NextResponse.redirect(url)
    }
  }

  // Authenticated route protection
  if (authRoutes.some(route => pathname.startsWith(route))) {
    const token = await getToken({ 
      req: request, 
      secret: process.env.NEXTAUTH_SECRET 
    })
    
    if (!token) {
      if (pathname.startsWith('/api/')) {
        return new NextResponse(
          JSON.stringify({ error: 'Unauthorized' }),
          { 
            status: 401, 
            headers: { 
              'Content-Type': 'application/json',
              ...Object.fromEntries(Object.entries(securityHeaders)),
            } 
          }
        )
      }
      
      const url = request.nextUrl.clone()
      url.pathname = '/auth/signin'
      url.searchParams.set('callbackUrl', pathname)
      return NextResponse.redirect(url)
    }
  }

  // API request validation
  if (pathname.startsWith('/api/') && !publicApiRoutes.some(route => pathname.startsWith(route))) {
    const validation = validateApiRequest(request)
    if (!validation.valid) {
      return new NextResponse(
        JSON.stringify({ error: validation.error }),
        { 
          status: validation.status,
          headers: { 
            'Content-Type': 'application/json',
            ...Object.fromEntries(Object.entries(securityHeaders)),
          }
        }
      )
    }
  }

  return response
}

async function applyRateLimit(request: NextRequest, pathname: string) {
  // Skip rate limiting in development mode
  if (process.env.NODE_ENV === 'development') {
    return {
      success: true,
      limit: 999,
      remaining: 999,
      resetTime: Date.now() + 60000,
    }
  }

  const ip = getClientIP(request)
  
  let config = rateLimitConfig.api
  
  if (pathname.startsWith('/api/auth')) {
    config = rateLimitConfig.auth
  } else if (pathname.startsWith('/api/payments')) {
    config = rateLimitConfig.payment
  } else if (pathname.includes('upload')) {
    config = rateLimitConfig.upload
  }
  
  return await rateLimit(ip, config.requests, config.window)
}

function validateApiRequest(request: NextRequest) {
  const contentType = request.headers.get('content-type')
  const method = request.method
  
  // Validate content type for POST/PUT requests
  if (['POST', 'PUT', 'PATCH'].includes(method)) {
    if (!contentType || !contentType.includes('application/json')) {
      return {
        valid: false,
        error: 'Invalid content type. Expected application/json',
        status: 400
      }
    }
  }
  
  // Validate request size (10MB limit)
  const contentLength = request.headers.get('content-length')
  if (contentLength && parseInt(contentLength) > 10 * 1024 * 1024) {
    return {
      valid: false,
      error: 'Request payload too large',
      status: 413
    }
  }
  
  return { valid: true }
}

export const config = {
  matcher: [
    /*
     * Match all request paths except:
     * - _next/static (static files)
     * - _next/image (image optimization files)
     * - favicon.ico (favicon file)
     * - public folder
     * - localhost (completely bypass for development)
     * - Frontend routes (e.g., '/', '/services', '/services/hosting', '/contact')
     */
    '/((?!_next/static|_next/image|favicon.ico|public/|$|services|services/hosting|contact).*)',
  ],
}