import { create } from 'zustand'
import { persist } from 'zustand/middleware'

export interface ServicePackage {
  id: string
  name: string
  price: number
  originalPrice?: number
  features: string[]
  description?: string
}

export interface ServiceAddon {
  id: string
  name: string
  price: number
  description?: string
  category?: string
}

export interface CartItem {
  id: string
  serviceId: string
  name: string
  price: number
  quantity: number
  selectedPackage?: ServicePackage
  configuration?: Record<string, any>
  addons?: ServiceAddon[]
  customizations?: Record<string, any>
  discount?: {
    type: 'percentage' | 'fixed'
    value: number
    reason: string
  }
}

export interface Bundle {
  id: string
  name: string
  description: string
  services: string[] // Service IDs
  discount: number // Percentage discount
  price: number
}

interface CartState {
  items: CartItem[]
  bundles: Bundle[]
  isOpen: boolean
  appliedPromoCode?: string
  promoDiscount?: number
  
  // Basic cart operations
  addItem: (item: Omit<CartItem, 'id'>) => void
  addToCart: (item: { id: number; title: string; price: number; quantity: number }) => void
  removeItem: (id: string) => void
  updateQuantity: (id: string, quantity: number) => void
  clearCart: () => void
  
  // Enhanced operations
  updatePackage: (itemId: string, selectedPackage: ServicePackage) => void
  addAddon: (itemId: string, addon: ServiceAddon) => void
  removeAddon: (itemId: string, addonId: string) => void
  updateCustomization: (itemId: string, customizations: Record<string, any>) => void
  applyPromoCode: (code: string) => boolean
  removePromoCode: () => void
  
  // Bundle operations
  addBundle: (bundle: Bundle) => void
  removeBundle: (bundleId: string) => void
  suggestBundles: () => Bundle[]
  
  // UI operations
  toggleCart: () => void
  openCart: () => void
  closeCart: () => void
  
  // Calculations
  getTotalItems: () => number
  getTotalPrice: () => number
  getBulkDiscount: () => number
  getTotalWithDiscounts: () => number
  getSavings: () => number
}

// Predefined bundles
const AVAILABLE_BUNDLES: Bundle[] = [
  {
    id: 'startup-bundle',
    name: 'Startup Essentials',
    description: 'Perfect for new businesses - Website + Social Media Marketing',
    services: ['1', '4'], // Website Development + Social Media Marketing
    discount: 15,
    price: 2550 // Original: 3000, with 15% discount
  },
  {
    id: 'marketing-bundle',
    name: 'Digital Marketing Suite',
    description: 'Complete marketing solution - Social Media + SEO + Digital Marketing',
    services: ['4', '5', '6'], // Social Media + SEO + Digital Marketing
    discount: 20,
    price: 2560 // Original: 3200, with 20% discount
  },
  {
    id: 'ecommerce-bundle',
    name: 'E-commerce Complete',
    description: 'Everything for online stores - E-commerce + SEO + Content',
    services: ['3', '5', '7'], // E-commerce + SEO + Content Writing
    discount: 18,
    price: 4100 // Original: 5000, with 18% discount
  }
]

// Bulk discount tiers
const BULK_DISCOUNT_TIERS = [
  { minItems: 2, discount: 5, label: '5% off 2+ services' },
  { minItems: 3, discount: 10, label: '10% off 3+ services' },
  { minItems: 5, discount: 15, label: '15% off 5+ services' }
]

// Promo codes
const PROMO_CODES: Record<string, number> = {
  'WELCOME10': 10,
  'SAVE15': 15,
  'NEWCLIENT': 20,
  'BUNDLE25': 25
}

export const useCartStore = create<CartState>()(
  persist(
    (set, get) => ({
      items: [],
      bundles: [],
      isOpen: false,
      appliedPromoCode: undefined,
      promoDiscount: undefined,
      
      addItem: (item) => {
        const items = get().items
        const existingItem = items.find(
          (i) => i.serviceId === item.serviceId && 
          JSON.stringify(i.configuration) === JSON.stringify(item.configuration) &&
          JSON.stringify(i.selectedPackage) === JSON.stringify(item.selectedPackage)
        )
        
        if (existingItem) {
          set({
            items: items.map((i) =>
              i.id === existingItem.id
                ? { ...i, quantity: i.quantity + item.quantity }
                : i
            ),
          })
        } else {
          const newItem: CartItem = {
            ...item,
            id: `${item.serviceId}_${Date.now()}_${Math.random()}`,
          }
          set({ items: [...items, newItem] })
        }
      },

      addToCart: (item) => {
        const items = get().items
        const serviceId = item.id.toString()
        const existingItem = items.find((i) => i.serviceId === serviceId)
        
        if (existingItem) {
          set({
            items: items.map((i) =>
              i.id === existingItem.id
                ? { ...i, quantity: i.quantity + item.quantity }
                : i
            ),
          })
        } else {
          const newItem: CartItem = {
            id: `${serviceId}_${Date.now()}_${Math.random()}`,
            serviceId,
            name: item.title,
            price: item.price,
            quantity: item.quantity,
          }
          set({ items: [...items, newItem] })
        }
      },
      
      removeItem: (id) => {
        set({ items: get().items.filter((item) => item.id !== id) })
      },
      
      updateQuantity: (id, quantity) => {
        if (quantity <= 0) {
          get().removeItem(id)
          return
        }
        
        set({
          items: get().items.map((item) =>
            item.id === id ? { ...item, quantity } : item
          ),
        })
      },
      
      clearCart: () => {
        set({ items: [], bundles: [], appliedPromoCode: undefined, promoDiscount: undefined })
      },

      // Enhanced operations
      updatePackage: (itemId, selectedPackage) => {
        set({
          items: get().items.map((item) =>
            item.id === itemId 
              ? { ...item, selectedPackage, price: selectedPackage.price }
              : item
          ),
        })
      },

      addAddon: (itemId, addon) => {
        set({
          items: get().items.map((item) =>
            item.id === itemId 
              ? { 
                  ...item, 
                  addons: [...(item.addons || []), addon]
                }
              : item
          ),
        })
      },

      removeAddon: (itemId, addonId) => {
        set({
          items: get().items.map((item) =>
            item.id === itemId 
              ? { 
                  ...item, 
                  addons: item.addons?.filter(addon => addon.id !== addonId) || []
                }
              : item
          ),
        })
      },

      updateCustomization: (itemId, customizations) => {
        set({
          items: get().items.map((item) =>
            item.id === itemId 
              ? { ...item, customizations: { ...item.customizations, ...customizations } }
              : item
          ),
        })
      },

      applyPromoCode: (code) => {
        const discount = PROMO_CODES[code.toUpperCase()]
        if (discount) {
          set({ appliedPromoCode: code.toUpperCase(), promoDiscount: discount })
          return true
        }
        return false
      },

      removePromoCode: () => {
        set({ appliedPromoCode: undefined, promoDiscount: undefined })
      },

      // Bundle operations
      addBundle: (bundle) => {
        const bundles = get().bundles
        if (!bundles.find(b => b.id === bundle.id)) {
          set({ bundles: [...bundles, bundle] })
        }
      },

      removeBundle: (bundleId) => {
        set({ bundles: get().bundles.filter(b => b.id !== bundleId) })
      },

      suggestBundles: () => {
        const items = get().items
        const serviceIds = items.map(item => item.serviceId)
        
        return AVAILABLE_BUNDLES.filter(bundle => {
          // Suggest bundles where at least one service is in cart but not all
          const bundleServices = bundle.services
          const hasAnyService = bundleServices.some(id => serviceIds.includes(id))
          const hasAllServices = bundleServices.every(id => serviceIds.includes(id))
          return hasAnyService && !hasAllServices
        })
      },

      // UI operations
      toggleCart: () => {
        set({ isOpen: !get().isOpen })
      },
      
      openCart: () => {
        set({ isOpen: true })
      },
      
      closeCart: () => {
        set({ isOpen: false })
      },
      
      // Calculations
      getTotalItems: () => {
        const items = get().items
        const bundles = get().bundles
        return items.reduce((total, item) => total + item.quantity, 0) + bundles.length
      },
      
      getTotalPrice: () => {
        const items = get().items
        const bundles = get().bundles
        
        const itemsTotal = items.reduce((total, item) => {
          const basePrice = (item.selectedPackage?.price || item.price) * item.quantity
          const addonsPrice = item.addons?.reduce((sum, addon) => sum + addon.price, 0) || 0
          const itemDiscount = item.discount ? 
            (item.discount.type === 'percentage' ? basePrice * (item.discount.value / 100) : item.discount.value) : 0
          return total + basePrice + addonsPrice - itemDiscount
        }, 0)

        const bundlesTotal = bundles.reduce((total, bundle) => total + bundle.price, 0)
        
        return itemsTotal + bundlesTotal
      },

      getBulkDiscount: () => {
        const totalItems = get().getTotalItems()
        const totalPrice = get().getTotalPrice()
        
        const applicableTier = BULK_DISCOUNT_TIERS
          .filter(tier => totalItems >= tier.minItems)
          .sort((a, b) => b.discount - a.discount)[0]
        
        return applicableTier ? (totalPrice * applicableTier.discount / 100) : 0
      },

      getTotalWithDiscounts: () => {
        const baseTotal = get().getTotalPrice()
        const bulkDiscount = get().getBulkDiscount()
        const promoDiscount = get().promoDiscount ? (baseTotal * get().promoDiscount! / 100) : 0
        
        return Math.max(0, baseTotal - bulkDiscount - promoDiscount)
      },

      getSavings: () => {
        const baseTotal = get().getTotalPrice()
        const finalTotal = get().getTotalWithDiscounts()
        return baseTotal - finalTotal
      },
    }),
    {
      name: 'cart-storage',
    }
  )
)